package presentation.graph;

/**
 * @author Alberto Iachini
 */

import org.jgraph.*;
import org.jgraph.graph.*;

import com.jgraph.layout.JGraphLayout;
import com.jgraph.layout.JGraphFacade;
import com.jgraph.layout.graph.JGraphSimpleLayout;
import com.jgraph.layout.hierarchical.JGraphHierarchicalLayout;
import com.jgraph.layout.organic.JGraphOrganicLayout;
import com.jgraph.layout.tree.JGraphTreeLayout;
import org.jgraph.util.ParallelEdgeRouter;

import abstraction.Service;
import abstraction.State;
import abstraction.Action;
import abstraction.StateAction;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.Vector;
import java.util.Iterator;

import java.awt.Color;
import java.awt.geom.Rectangle2D;
import java.awt.geom.Point2D;

/**
 * Factory to create a JGraph object from a Service
 */
public class GraphFactory {
	
	private JGraph graph;
	private HashMap<State, DefaultGraphCell> vertexMap;
	private HashMap<StateAction, DefaultEdge> edgeMap;
	
	/**
	 * Create a new GraphFactory
	 * @param service the service
	 */
	public GraphFactory(Service service)
	{
		createGraph(service);
	}
	
	/**
	 * return the generated graph
	 * @return the graph
	 */
	public JGraph getGraph() {
		return graph;
	}
	
	/**
	 * Return an hash map
	 * 	key: state of service
	 * 	value: vertex of graph
	 * @return the map <State,DefaultGraphCell>
	 */
	public HashMap<State, DefaultGraphCell> getVertexMap() {
		return vertexMap;
	}

	/**
	 * Return an hash map
	 * 	key: action of service
	 * 	value: edge of graph
	 * @return  the map <Action, DefaultEdge>
	 */
	public HashMap<StateAction, DefaultEdge> getEdgeMap() {
		return edgeMap;
	}

	/**
	 * Create a JGraph component from service
	 * @param service the selected service 
	 */
	private void createGraph(Service service)
	{
		vertexMap = new HashMap<State, DefaultGraphCell>();
		edgeMap = new HashMap<StateAction, DefaultEdge>();
		
		// vector that holds the cell to show
		Vector<DefaultGraphCell> cells = new Vector<DefaultGraphCell>();
		
		// temporary hash table that holds the vertexes already created
		HashMap<String, DefaultGraphCell> temp = new HashMap<String, DefaultGraphCell>();
		
		// iterator on nodes of the service
		Iterator<State> nodes = service.getStates();
		
		while(nodes.hasNext())
		{
			// name of node
			State node = nodes.next();
			String nodeName = node.getName();
			DefaultGraphCell source;
			if(temp.containsKey(nodeName))
			{
				source = temp.get(nodeName);
			}
			else
			{
				source = createVertex(nodeName,service.isFinalState(node));
				temp.put(nodeName, source);
				cells.add(source);
				
				vertexMap.put(node, source);
				
			}
			
			//iterator on actions of the node
			Iterator<Action> edges = service.getActions(node);
			while(edges.hasNext())
			{
				Action act = edges.next();
				String edge = act.getName();
				//iterator on target nodes of the edges that have the same action 
				Iterator<State> targetNodes = service.getNextStates(node, act);
				while(targetNodes.hasNext())
				{
					State targetNode = targetNodes.next();
					String targetNodeName = targetNode.getName();
					DefaultGraphCell dest = null;
					if(temp.containsKey(targetNodeName))
					{
						dest = temp.get(targetNodeName);
					}
					else
					{
						dest = createVertex(targetNodeName, service.isFinalState(targetNode));
						temp.put(targetNodeName, dest);
						cells.add(dest);
						
						vertexMap.put(targetNode, dest);
					}
					DefaultEdge edgeCell = createEdge(edge, source, dest);
					cells.add(edgeCell);
					
					edgeMap.put(new StateAction(node,act), edgeCell);
				}
			}
			
			
		}
		// create a new JGrapf object
		GraphModel model = new DefaultGraphModel();
		GraphLayoutCache view = new GraphLayoutCache(model, 
				new StateCellViewFactory());
		
		graph = new JGraph(model, view);
		
		graph.setEditable(false);
		graph.setConnectable(false);
		graph.setDisconnectable(false);
		graph.setAntiAliased(true);
		
		// add the cells into graph
		graph.getGraphLayoutCache().insert(cells.toArray());
		
		changeRepresentation(graph,layouts[0]);
	}
	
	/**
	 * create a vertex
	 * @param node the name of the vertex
	 * @param isFinalState true if node is final
	 * @return the vertex
	 */
	public static DefaultGraphCell createVertex(String node, boolean isFinalState)
	{
		DefaultGraphCell cell;
		if(isFinalState==true)
		{
			//create a vertex with double circle shape (final state)
			cell = new FinalStateVertex(node);
		}
		else
		{
			////create a vertex with single circle shape (not final state)
			cell = new DefaultGraphCell(node);
		}
		
		
		GraphConstants.setBounds(cell.getAttributes(), new Rectangle2D.Double(10,10,100,60));
		GraphConstants.setGradientColor(cell.getAttributes(), Color.BLACK);
		GraphConstants.setOpaque(cell.getAttributes(), true);
		// add port
		cell.add(new DefaultPort());
		
		return cell;
	}
	
	/**
	 * create an edge
	 * @param edge the name of the action
	 * @param source source vertex
	 * @param target target vertex
	 * @return the edge
	 */
	public static DefaultEdge createEdge(String edge, DefaultGraphCell source, DefaultGraphCell target)
	{
		// create a default edge
		DefaultEdge cell = new DefaultEdge(edge);
		
		// set source and target
		cell.setSource(source.getChildAt(0));
		cell.setTarget(target.getChildAt(0));
		
		GraphConstants.setLineEnd(cell.getAttributes(), GraphConstants.ARROW_CLASSIC);
		GraphConstants.setEndFill(cell.getAttributes(), true);
		Point2D position = new Point2D.Double(GraphConstants.PERMILLE*2/5, -10);
		GraphConstants.setLabelPosition(cell.getAttributes(), position);
		return cell;
	}
	
	/**
	 * Change the color of the vertex
	 * @param graph the graph
	 * @param cell the selected vertex
	 * @param select the vertex is blue if true or it is black if false
	 */
	@SuppressWarnings("unchecked")
	public static void selectVertex(JGraph graph,  DefaultGraphCell cell, boolean select)
	{
		Color color;
		if(select == true)
			color = Color.BLUE;
		else
			color = Color.BLACK;
				
		Hashtable attrMap = new Hashtable();
		GraphConstants.setGradientColor(attrMap, color);
		
		Hashtable nested = new Hashtable();
		nested.put(cell, attrMap);
		
		graph.getGraphLayoutCache().edit(nested);
	}
	
	/**
	 * Change the color of the edge
	 * @param graph the graph
	 * @param edge the selected edge
	 * @param select the edge is red if true or black if false
	 */
	@SuppressWarnings("unchecked")
	public static void selectEdge(JGraph graph, DefaultEdge edge, boolean select)
	{
		Color color;
		if(select == true)
			color = Color.RED;
		else 
			color = Color.BLACK;
		
		Hashtable attrMap = new Hashtable();
		GraphConstants.setLineColor(attrMap, color);
		
		Hashtable nested = new Hashtable();
		nested.put(edge, attrMap);
		
		graph.getGraphLayoutCache().edit(nested);
	}
	
	/**
	 * Change the layout of the graph
	 * @param graph the graph
	 * @param layout the name of the layout
	 */
	public static void changeRepresentation(JGraph graph, String layout)
	{
		ParallelEdgeRouter.setEdgeSeparation(20);
		ParallelEdgeRouter.setEdgeDeparture(50);
		
		JGraphFacade fac = new JGraphFacade(graph);
		JGraphLayout lay;	
		fac.resetControlPoints(true, ParallelEdgeRouter.getSharedInstance());
		
		
		if(layout.equals(layouts[0]))
		{
			//hierarchical
			lay = new JGraphHierarchicalLayout();
		}
		else if(layout.equals(layouts[1]))
		{
			//circle
			lay = new JGraphSimpleLayout(JGraphSimpleLayout.TYPE_CIRCLE);
		}
		else if(layout.equals(layouts[2]))
		{
			// tree
			lay = new JGraphTreeLayout();
		}
		else//if(layout.equals(layouts[3]))
		{
			// organic
			lay = new JGraphOrganicLayout();
		}
		
		lay.run(fac);
		graph.getGraphLayoutCache().edit(fac.createNestedMap(true,true));
	}
	public static final String[] layouts = {"Hierarchical", "Circle",
		"Tree", "Organic"};

}
